/* Copyright (C) 2014-2018 RealVNC Ltd.  All Rights Reserved.
*/

#ifndef UUID_a377ccf1_d6e0_4ad9_ac55_470ff369f7eb
#define UUID_a377ccf1_d6e0_4ad9_ac55_470ff369f7eb

/**
 * \file vnccommontypes.h
 *
 * This header defines common types used by several of the VNC SDKs.
 *
 * The vncint.h header is included below. See the vncint.h documentation for
 * information on how to instruct the SDK to define integer types.
 */

#include "vncint.h"

#ifdef __cplusplus
extern "C"
{
#endif


/**
 * \brief Describes a point in two dimensions.
 *
 * The origin, (0, 0), is the point at the top left of the framebuffer.
 *
 * \see VNCRectangle
 */
typedef struct
{
    /** The X-coordinate of the point. */
    vnc_int16_t x;
    /** The Y-coordinate of the point. */
    vnc_int16_t y;
} VNCPoint;

/**
 * \brief Describes a rectangular region.
 *
 * The rectangle contains the pixels from topLeft inclusive through to
 * bottomRight non-inclusive.  That is, there are
 * (bottomRight.x - topLeft.x) * (bottomRight.y - topLeft.y) pixels in the
 * rectangle.
 *
 * \see VNCPoint
 */
typedef struct
{
    /** The point inside the rectangle at its top-left corner. */
    VNCPoint topLeft;
    /** The point just outside the rectangle at its bottom-right corner. */
    VNCPoint bottomRight;
} VNCRectangle;

/**
 * \brief Bitmask values indicating different buttons on a pointer device.
 *
 * \if VIEWER_SDK
 * RFB uses values of the form (2 << (N - 1)) to indicate 'pointer device
 * button N).  This enumeration provides symbolic names for these values.
 * \endif
 *
 * On a conventional mouse, buttons 1, 2 and 3 correspond to the left, middle
 * and right buttons on the mouse respectively. On a wheel mouse, each step of
 * the wheel upwards is represented by a press and release of button 4, and
 * each step downwards is represented by a press and release of button 5.
 *
 * \if VIEWER_SDK
 * \see VNCViewerSendPointerEvent
 * \endif
 */
typedef enum
{
    /* \cond */
    VNCPointerDeviceButtonNone = 0x00,
    VNCPointerDeviceButton1 = 0x01,
    VNCPointerDeviceButton2 = 0x02,
    VNCPointerDeviceButton3 = 0x04,
    VNCPointerDeviceButton4 = 0x08,
    VNCPointerDeviceButton5 = 0x10,
    VNCPointerDeviceButton6 = 0x20,
    VNCPointerDeviceButton7 = 0x40,
    VNCPointerDeviceButton8 = 0x80,

    VNCPointerDeviceButtonLeft = VNCPointerDeviceButton1,
    VNCPointerDeviceButtonMiddle = VNCPointerDeviceButton2,
    VNCPointerDeviceButtonRight = VNCPointerDeviceButton3,
    VNCPointerDeviceButtonWheelUp = VNCPointerDeviceButton4,
    VNCPointerDeviceButtonWheelDown = VNCPointerDeviceButton5
    /* \endcond */
} VNCPointerDeviceButton;

/**
 * \brief Describes a single touch in a multi-touch event.
 *
 * \if VIEWER_SDK
 * This structure is for use with VNCViewerSendTouchEvent(). An array of these
 * structure is used to communicate multiple points of contact to the VNC
 * Server.
 *
 * \see VNCViewerSendTouchEvent
 * \endif
 */
typedef struct
{
    /**
     * \brief The location of this point of contact within the framebuffer.
     */
    VNCPoint location;

    /**
     * \brief An identifier for this point of contact.
     *
     * This identifier should be repeated in subsequent SDK API calls to
     * indicate that the same point of contact is being maintained. This allows
     * the server to track a particular point of contact (i.e. a finger), even
     * as it moves.
     *
     * If it is not possible to identify and track points of contact
     * between SDK API calls  then this should be set to
     * #VNCTouchDescriptorUnidentified.
     *
     * \if VIEWER_SDK
     * In MirrorLink sessions this value must be between 0 and the
     * supported touch count value returned in the pointerSupport
     * field in the VNCServerEventConfiguration structure.
     * \endif
     * 
     * A value of VNCTouchDescriptorUnidentified can not be used in MirrorLink
     * sessions.
     *
     * /see VNCTouchDescriptorUnidentified
     */
    vnc_uint8_t identifier;

    /**
     * \brief The pressure that is being applied at this point of contact.
     *
     * In non-MirrorLink sessions, pressure values should range from 0 to 255
     * inclusive, where 255 is the maximum detectable pressure.
     *
     * \if VIEWER_SDK
     * In MirrorLink sessions, the application must supply its maximum pressure
     * value in the pointerSupport member of the VNCClientEventConfiguration
     * message. In this case, pressure values should range from 0 to the
     * supplied maximum value. It is the application's responsibility to ensure
     * that the pressure values that it supplies lie within the correct range.
     *
     * In both cases, a pressure of 0 indicates that the point of contact is no
     * longer being touched.
     * \endif
     */
    vnc_uint8_t pressure;

    /**
     * \brief A bit-mask representing the buttons that are currently depressed.
     *
     * This would normally apply to styluses, as they sometimes have hardware
     * buttons. If this VNCTouchDescriptor represents a finger, or a device
     * that doesn't have any buttons, then this field should be set to 
     * VNCPointerDeviceButtonLeft.
     */
    VNCPointerDeviceButton buttonMask;
} VNCTouchDescriptor;

/**
 * \brief For touch events where individual points of contact can not be
 * identified.
 *
 * Used as the identifier field in a VNCTouchDescriptor when points of contact
 * can not be individually identified and tracked. This might be due to
 * limitations in the hardware or software environment for the platform
 * running the viewer.
 *
 * \see VNCTouchDescriptor
 */
#define VNCTouchDescriptorUnidentified 255

/**
 * \brief Abstracted button codes.
 *
 * \if VIEWER_SDK
 * These values are for use with VNCViewerSendDeviceKeyEvent().
 * \endif
 *
 * These abstractions represent buttons that are present on the majority of
 * mobile devices, and, where they are present, are necessary to perform the
 * majority of remote support tasks.  Buttons that are present only on a
 * minority of devices, or which have functions that can be carried out by
 * other means, are deliberately not included.
 *
 * The number keys in this enumeration represent number buttons on a telephone
 * keypad.  These are distinct from the X key symbols representing numbers on
 * the top row of a US keyboard (e.g. XK_1) and those representing numbers on a
 * PC keyboard's numeric keypad (e.g. XK_KP_1).
 *
 * For VNC purposes, trackballs on mobile devices are not considered to be
 * pointer devices.  This is because they typically do not freely move a 
 * pointer across the whole range of the display, but instead move an input
 * cursor along a one- or two-dimensional grid.  They are therefore considered
 * equivalent to direction keys.
 *
 * \if VIEWER_SDK
 * \see VNCSendDeviceKeyEvent().
 * \endif
 */
typedef enum
{
    /** A cellphone handset's left soft key. */
    VNCDeviceKeyLeftSoftKey = 0,

    /** A cellphone handset's right soft key. */
    VNCDeviceKeyRightSoftKey = 1,

    /** A cellphone handset's send (make voice call) button. */
    VNCDeviceKeySend = 2,

    /** A cellphone handset's end (hangup) button. */
    VNCDeviceKeyEnd = 3,

    /** A cellphone handset's volume up button. */
    VNCDeviceKeyVolumeUp = '+',

    /** A cellphone handset's volume down button. */
    VNCDeviceKeyVolumeDown = '-',

    /** The 0 key on a cellphone handset's phone keypad. */
    VNCDeviceKeyKeypad0 = '0',

    /** The 1 key on a cellphone handset's phone keypad. */
    VNCDeviceKeyKeypad1 = '1',

    /** The 2 key on a cellphone handset's phone keypad. */
    VNCDeviceKeyKeypad2 = '2',

    /** The 3 key on a cellphone handset's phone keypad. */
    VNCDeviceKeyKeypad3 = '3',

    /** The 4 key on a cellphone handset's phone keypad. */
    VNCDeviceKeyKeypad4 = '4',

    /** The 5 key on a cellphone handset's phone keypad. */
    VNCDeviceKeyKeypad5 = '5',

    /** The 6 key on a cellphone handset's phone keypad. */
    VNCDeviceKeyKeypad6 = '6',

    /** The 7 key on a cellphone handset's phone keypad. */
    VNCDeviceKeyKeypad7 = '7',

    /** The 8 key on a cellphone handset's phone keypad. */
    VNCDeviceKeyKeypad8 = '8',

    /** The 9 key on a cellphone handset's phone keypad. */
    VNCDeviceKeyKeypad9 = '9',

    /** The star key on a cellphone handset's phone keypad. */
    VNCDeviceKeyKeypadStar = '*',

    /** The pound sign (hash) key on a cellphone handset's phone keypad. */
    VNCDeviceKeyKeypadPoundSign = '#',

    /**
     * \brief The left directional button, or a roll of the trackball to the
     * left.
     */
    VNCDeviceKeyLeft = 'h',

    /**
     * \brief The right directional button, or a roll of the trackball to the
     * right.
     */
    VNCDeviceKeyRight = 'l',

    /**
     * \brief The up directional button, or a roll of the trackball upwards.
     */
    VNCDeviceKeyUp = 'k',

    /**
     * \brief The down directional button, or a roll of the trackball
     * downwards.
     */
    VNCDeviceKeyDown = 'j',

    /**
     * \brief An abstraction for a device button that selects an item from a
     * menu.
     *
     * This abstraction represents these buttons on these platforms:
     *
     * - Android - the 'D-pad center' key (on devices with a trackball, this is
     *   a trackball click)
     */
    VNCDeviceKeySelect = 'S',

    /**
     * \brief An abstraction for a device button that cancels an action or
     * navigates backwards.
     *
     * This abstraction represents these buttons on these platforms:
     *
     * - Android - the 'back' button
     */
    VNCDeviceKeyDismiss = 'D',

    /**
     * \brief An abstraction for a device button that navigates backwards.
     *
     * This abstraction represents these buttons on these platforms:
     *
     * - Android - the 'back' button
     */
    VNCDeviceKeyBack = 'B',

    /**
     * \brief An abstraction for a device button that returns the device to its
     * homescreen.
     *
     * This is indentical to VNCDeviceKeyMenu, except on those server platforms
     * that typically make a distinction (e.g. Android).
     *
     * This abstraction represents these buttons on these platforms:
     *
     * - Android - the 'home' button
     */
    VNCDeviceKeyHome = 'H',

    /**
     * \brief An abstraction for a device button that opens the device's main
     * menu.
     *
     * This is indentical to VNCDeviceKeyMenu, except on those server platforms
     * that typically make a distinction (e.g. Android).
     *
     * This abstraction represents these buttons on these platforms:
     *
     * - Android - the 'menu' button
     */
    VNCDeviceKeyMenu = 'M',

    /**
     * \brief An abstraction for a device button that unlocks a locked phone
     * keypad.
     *
     * If the VNC Server host device has a button that whose function is to
     * unlock the keypad, then this abstraction maps to that button.  VNC
     *
     * Servers on other devices will ignore this abstraction.
     */
    VNCDeviceKeyUnlock = 'U',

    /**
     * \brief An abstraction for a device button that changes the text input
     * mode.
     */
    VNCDeviceKeyEdit = 'E',

    /** A cellphone handset's power button. */
    VNCDeviceKeyPower = 'P',
} VNCDeviceKey;

/**
 * \brief Defines the format of each pixel in a framebuffer.
 *
 * \if VIEWER_SDK
 * A VNC session begins by using the server's native pixel format.  The
 * application is notified of this via its VNCViewerServerInitCallback().
 * However, the application may choose a different pixel format by calling
 * VNCViewerSetPixelFormat().
 *
 * You should choose custom pixel formats carefully so that it is easy for the
 * VNC Server to generate pixel data in that format, and easy for your
 * application to render.  If processing power on the VNC Server host is
 * limited, it may be best to continue to use the server's native pixel format
 * for the session.  Conversely, if the network is slow or expensive to use, it
 * may be necessary to switch to a more compact pixel format.
 * \endif
 *
 * \if HSML_SDK
 * You should use this structure to describe the pixel format of your
 * application's framebuffer when calling VNCHSMLSinkCreate(). If this pixel
 * format differs from the pixel format used for HSML session, then the SDK
 * shall perform the necessary conversion.
 * \endif
 *
 * \section byte_order Byte Order
 *
 * Pixel data can be big-endian or little-endian.  If the byte order differs
 * from the byte order that is used by your application's framebuffer, then you
 * must swap the byte order of every pixel before further interpreting it.
 *
 * \section true_color Using True Color
 *
 * If trueColorFlag is non-zero, then the red, green and blue components of
 * each pixel are contained directly in the pixel data.  Each component can be
 * extracted using the maximum and shift values for that component.
 *
 * The maximum value for each component is (2^n - 1), where n is the number of
 * bits used to store that component.  For example, if a pixel's red component
 * is equal to redMax, then its red component is at full intensity.
 *
 * The shift value for each component is the number of right-shifts that must
 * be applied to each pixel value to move the least-significant-bit of that
 * component component into the least-significant-bit of the result.  For
 * example, after dealing with the pixel byte order, the red component can be
 * extracted by calculating ((pixel >> redShift) & redMax).
 *
 * For example, if bitsPerPixel is 8 and each pixel has the format 00rrggbb,
 * then:
 *
 *  - redMax, greenMax and blueMax are all equal to 3
 *  - redShift is 4
 *  - blueShift is 2
 *  - greenShift is 0
 *
 * \section color_map Using an Indexed Pixel Format
 *
 * If trueColorFlag is zero, then the pixel format is said to be 'indexed'.
 * This means that each value in the pixel data is an index into a color map
 * chosen by the VNC Server.  Indexed pixel formats often look better than true
 * color pixel formats when the number of distinct pixel values is small.
 *
 * \if VIEWER_SDK
 * \see VNCViewerCreate, VNCViewerLockRectangleExCallback,
 * VNCViewerUnlockRectangleExCallback, VNCViewerSetPixelFormat
 * \endif
 *
 * \if HSML_SDK
 * \see VNCHSMLSinkCreate
 * \endif
 */
typedef struct
{
    /**
     * The number of bits of pixel data used for each pixel.  Must be either 8,
     * 16, or 32.
     *
     * Your application's framebuffer must be aligned so that the address of
     * each pixel is a multiple of (bitsPerPixel / 8).
     */
    vnc_uint8_t bitsPerPixel;

    /**
     * The number of bits in each pixel that are significant (i.e. the number
     * of colors that can be represented).  Must be less than or equal to
     * bitsPerPixel.  (For example, if 32 bits are used to represent pixels
     * that have 8 bits each for red, green and blue, then bitsPerPixel is 32
     * and depth is 24.)
     *
     * Non-signficiant bits are cleared by the SDK.
     */
    vnc_uint8_t depth;

    /**
     * If non-zero, then the pixels are big-endian.  Otherwise, they are
     * little-endian.
     */
    vnc_uint8_t bigEndianFlag;

    /**
     * If non-zero, then the pixel data contains actual pixel values.
     * Otherwise, the pixel data contains indices into the accompanying color
     * map.
     */
    vnc_uint8_t trueColorFlag;

    /**
     * The maximum value of the red component of each pixel.
     */
    vnc_uint16_t redMax;

    /**
     * The maximum value of the green component of each pixel.
     */
    vnc_uint16_t greenMax;

    /**
     * The maximum value of the blue component of each pixel.
     */
    vnc_uint16_t blueMax;

    /**
     * The shift value of the red component of each pixel.
     */
    vnc_uint8_t redShift;

    /**
     * The shift value of the green component of each pixel.
     */
    vnc_uint8_t greenShift;

    /**
     * The shift value of the blue component of each pixel.
     */
    vnc_uint8_t blueShift;
} VNCPixelFormat;

#ifdef __cplusplus
}
#endif

#endif /* !defined(UUID_a377ccf1_d6e0_4ad9_ac55_470ff369f7eb) */
